<?php
/**
 * Plugin Name: My Discogs Discography Pro
 * Description: Komplette Diskographie-Verwaltung mit Discogs-Integration und manuellen Einträgen.
 * Version: 2.4
 * Author: Marco Dinkel & Claude
 */

defined('ABSPATH') || exit;

class MyDC_Discography_Pro {

    private $post_type = 'mydc_release';

    public function __construct() {
        add_action('init', [$this, 'register_post_type']);
        add_action('add_meta_boxes', [$this, 'add_meta_boxes']);
        add_action('save_post', [$this, 'save_meta']);
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'admin_scripts']);
        add_action('wp_ajax_mydc_fetch_discogs_release', [$this, 'ajax_fetch_discogs_release']);
        add_action('wp_ajax_mydc_get_release_details', [$this, 'ajax_get_release_details']);
        add_action('wp_ajax_nopriv_mydc_get_release_details', [$this, 'ajax_get_release_details']);
        add_shortcode('mydc_discography', [$this, 'render_discography']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    public function register_post_type() {
        register_post_type($this->post_type, [
            'labels' => [
                'name' => 'Releases',
                'singular_name' => 'Release',
                'add_new' => 'Neues Release',
                'add_new_item' => 'Neues Release hinzufügen',
                'edit_item' => 'Release bearbeiten',
                'menu_name' => 'My Discogs Discography'
            ],
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => true,
            'menu_icon' => 'dashicons-album',
            'supports' => ['title'],
            'capability_type' => 'post'
        ]);

        register_taxonomy('mydc_release_category', $this->post_type, [
            'labels' => [
                'name' => 'Kategorien',
                'singular_name' => 'Kategorie',
                'add_new_item' => 'Neue Kategorie',
            ],
            'hierarchical' => true,
            'show_admin_column' => true,
            'public' => false,
            'show_ui' => true
        ]);
    }

    public function add_meta_boxes() {
        add_meta_box('mydc_release_details', 'Release Details', [$this, 'render_meta_box'], $this->post_type, 'normal', 'high');
    }

    public function render_meta_box($post) {
        wp_nonce_field('mydc_release_meta', 'mydc_release_nonce');
        $meta = get_post_meta($post->ID);
        $source = $meta['_mydc_source'][0] ?? 'manual';
        $discogs_id = $meta['_mydc_discogs_id'][0] ?? '';
        $discogs_type = $meta['_mydc_discogs_type'][0] ?? 'release';
        $artist = $meta['_mydc_artist'][0] ?? '';
        $year = $meta['_mydc_year'][0] ?? '';
        $label = $meta['_mydc_label'][0] ?? '';
        $format = $meta['_mydc_format'][0] ?? '';
        $genre = $meta['_mydc_genre'][0] ?? '';
        $cover_url = $meta['_mydc_cover_url'][0] ?? '';
        $discogs_url = $meta['_mydc_discogs_url'][0] ?? '';
        $tracklist = $meta['_mydc_tracklist'][0] ?? '';
        $role = $meta['_mydc_role'][0] ?? '';
        ?>
        <div class="mydc-meta-box">
            <table class="form-table">
                <tr>
                    <th>Quelle</th>
                    <td>
                        <label><input type="radio" name="mydc_source" value="discogs" <?php checked($source, 'discogs'); ?>> Von Discogs laden</label><br>
                        <label><input type="radio" name="mydc_source" value="manual" <?php checked($source, 'manual'); ?>> Manuell eingeben</label>
                    </td>
                </tr>
            </table>
            <div id="mydc-discogs-section" style="<?php echo $source !== 'discogs' ? 'display:none;' : ''; ?>">
                <table class="form-table">
                    <tr>
                        <th>Discogs Release-ID</th>
                        <td>
                            <input type="text" name="mydc_discogs_id" value="<?php echo esc_attr($discogs_id); ?>" class="regular-text" id="mydc-discogs-id">
                            <select name="mydc_discogs_type" id="mydc-discogs-type">
                                <option value="release" <?php selected($discogs_type, 'release'); ?>>Release</option>
                                <option value="master" <?php selected($discogs_type, 'master'); ?>>Master</option>
                            </select>
                            <button type="button" class="button" id="mydc-fetch-discogs">Von Discogs laden</button>
                            <p class="description">ID aus der Discogs-URL, z.B. 8401988 aus /release/8401988-...</p>
                            <div id="mydc-discogs-status"></div>
                        </td>
                    </tr>
                </table>
            </div>
            <div id="mydc-manual-section">
                <table class="form-table">
                    <tr><th>Künstler</th><td><input type="text" name="mydc_artist" value="<?php echo esc_attr($artist); ?>" class="large-text" id="mydc-artist"></td></tr>
                    <tr><th>Jahr</th><td><input type="text" name="mydc_year" value="<?php echo esc_attr($year); ?>" class="small-text" id="mydc-year"></td></tr>
                    <tr><th>Label</th><td><input type="text" name="mydc_label" value="<?php echo esc_attr($label); ?>" class="regular-text" id="mydc-label"></td></tr>
                    <tr><th>Format</th><td><input type="text" name="mydc_format" value="<?php echo esc_attr($format); ?>" class="regular-text" id="mydc-format" placeholder="z.B. CD, Vinyl, Digital"></td></tr>
                    <tr><th>Genre</th><td><input type="text" name="mydc_genre" value="<?php echo esc_attr($genre); ?>" class="regular-text" id="mydc-genre"></td></tr>
                    <tr><th>Rolle</th><td><input type="text" name="mydc_role" value="<?php echo esc_attr($role); ?>" class="regular-text" id="mydc-role" placeholder="z.B. Main, Appearance, Gastmusiker"><p class="description">Für die Filterung im Frontend</p></td></tr>
                    <tr><th>Cover URL</th><td><input type="url" name="mydc_cover_url" value="<?php echo esc_url($cover_url); ?>" class="large-text" id="mydc-cover-url"><?php if ($cover_url): ?><br><img src="<?php echo esc_url($cover_url); ?>" style="max-width: 150px; margin-top: 10px;"><?php endif; ?></td></tr>
                    <tr><th>Discogs URL</th><td><input type="url" name="mydc_discogs_url" value="<?php echo esc_url($discogs_url); ?>" class="large-text" id="mydc-discogs-url" placeholder="https://www.discogs.com/release/..."></td></tr>
                    <tr><th>Tracklist</th><td><textarea name="mydc_tracklist" rows="8" class="large-text" id="mydc-tracklist" placeholder="Ein Track pro Zeile, optional mit Dauer: Trackname (3:45)"><?php echo esc_textarea($tracklist); ?></textarea></td></tr>
                </table>
            </div>
        </div>
        <?php
    }

    public function save_meta($post_id) {
        if (!isset($_POST['mydc_release_nonce']) || !wp_verify_nonce($_POST['mydc_release_nonce'], 'mydc_release_meta')) return;
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        if (!current_user_can('edit_post', $post_id)) return;
        $fields = ['_mydc_source'=>'mydc_source','_mydc_discogs_id'=>'mydc_discogs_id','_mydc_discogs_type'=>'mydc_discogs_type','_mydc_artist'=>'mydc_artist','_mydc_year'=>'mydc_year','_mydc_label'=>'mydc_label','_mydc_format'=>'mydc_format','_mydc_genre'=>'mydc_genre','_mydc_cover_url'=>'mydc_cover_url','_mydc_discogs_url'=>'mydc_discogs_url','_mydc_tracklist'=>'mydc_tracklist','_mydc_role'=>'mydc_role'];
        foreach ($fields as $meta_key => $post_key) {
            if (isset($_POST[$post_key])) update_post_meta($post_id, $meta_key, sanitize_textarea_field($_POST[$post_key]));
        }
    }

    public function admin_scripts($hook) {
        global $post_type;
        if ($post_type !== $this->post_type) return;
        wp_add_inline_script('jquery', '
jQuery(document).ready(function($) {
    $("input[name=mydc_source]").on("change", function() {
        if ($(this).val() === "discogs") { $("#mydc-discogs-section").show(); } else { $("#mydc-discogs-section").hide(); }
    });
    $("#mydc-fetch-discogs").on("click", function() {
        var id = $("#mydc-discogs-id").val();
        var type = $("#mydc-discogs-type").val();
        if (!id) { alert("Bitte Release-ID eingeben"); return; }
        $("#mydc-discogs-status").html("<em>Lade von Discogs...</em>");
        $.post(ajaxurl, { action: "mydc_fetch_discogs_release", release_id: id, type: type }, function(response) {
            if (response.success) {
                var d = response.data;
                $("#title").val(d.title || "");
                if (d.artists) $("#mydc-artist").val(d.artists.map(a => a.name).join(", "));
                if (d.year) $("#mydc-year").val(d.year);
                if (d.labels && d.labels[0]) $("#mydc-label").val(d.labels[0].name);
                if (d.formats && d.formats[0]) { var fmt = d.formats[0].name; if (d.formats[0].descriptions) fmt += " (" + d.formats[0].descriptions.join(", ") + ")"; $("#mydc-format").val(fmt); }
                if (d.genres) $("#mydc-genre").val(d.genres.join(", "));
                if (d.images && d.images[0]) $("#mydc-cover-url").val(d.images[0].uri);
                if (d.uri) $("#mydc-discogs-url").val(d.uri);
                if (d.tracklist) { var tracks = d.tracklist.filter(t => t.type_ === "track").map(t => t.title + (t.duration ? " (" + t.duration + ")" : "")).join("\n"); $("#mydc-tracklist").val(tracks); }
                $("#mydc-discogs-status").html("<span style=\"color:green;\">Daten geladen!</span>");
            } else { $("#mydc-discogs-status").html("<span style=\"color:red;\">Fehler: " + response.data + "</span>"); }
        });
    });
});');
    }

    public function add_settings_page() {
        add_submenu_page('edit.php?post_type=' . $this->post_type, 'Einstellungen', 'Einstellungen', 'manage_options', 'mydc-discography-settings', [$this, 'render_settings_page']);
    }

    public function register_settings() {
        register_setting('mydc_discography_settings', 'mydc_discography_options');
        add_settings_section('mydc_api_section', 'Discogs API', null, 'mydc-discography-settings');
        add_settings_field('api_token', 'API Token', [$this, 'render_token_field'], 'mydc-discography-settings', 'mydc_api_section');
        add_settings_field('artist_id', 'Artist ID (für Auto-Import)', [$this, 'render_artist_id_field'], 'mydc-discography-settings', 'mydc_api_section');
        add_settings_section('mydc_design_section', 'Design Einstellungen', null, 'mydc-discography-settings');
        add_settings_field('color_card_bg', 'Kachel Hintergrund', [$this, 'render_color_card_bg'], 'mydc-discography-settings', 'mydc_design_section');
        add_settings_field('color_card_title', 'Titel Farbe', [$this, 'render_color_card_title'], 'mydc-discography-settings', 'mydc_design_section');
        add_settings_field('color_card_text', 'Text Farbe (Jahr, Rolle)', [$this, 'render_color_card_text'], 'mydc-discography-settings', 'mydc_design_section');
        add_settings_field('color_card_hover', 'Hover Schatten', [$this, 'render_color_card_hover'], 'mydc-discography-settings', 'mydc_design_section');
        add_settings_field('color_nocover_bg', 'Kein-Cover Hintergrund', [$this, 'render_color_nocover_bg'], 'mydc-discography-settings', 'mydc_design_section');
        add_settings_field('color_nocover_text', 'Kein-Cover Text', [$this, 'render_color_nocover_text'], 'mydc-discography-settings', 'mydc_design_section');
        add_settings_section('mydc_modal_section', 'Modal Design', null, 'mydc-discography-settings');
        add_settings_field('color_modal_bg', 'Modal Hintergrund', [$this, 'render_color_modal_bg'], 'mydc-discography-settings', 'mydc_modal_section');
        add_settings_field('color_modal_text', 'Modal Text', [$this, 'render_color_modal_text'], 'mydc-discography-settings', 'mydc_modal_section');
        add_settings_field('color_modal_meta', 'Modal Meta-Text', [$this, 'render_color_modal_meta'], 'mydc-discography-settings', 'mydc_modal_section');
        add_settings_field('color_modal_btn_bg', 'Discogs-Button Hintergrund', [$this, 'render_color_modal_btn_bg'], 'mydc-discography-settings', 'mydc_modal_section');
        add_settings_field('color_modal_btn_text', 'Discogs-Button Text', [$this, 'render_color_modal_btn_text'], 'mydc-discography-settings', 'mydc_modal_section');
    }

    public function render_token_field() { $options = get_option('mydc_discography_options'); echo '<input type="password" name="mydc_discography_options[api_token]" value="' . esc_attr($options['api_token'] ?? '') . '" class="regular-text">'; }
    public function render_artist_id_field() { $options = get_option('mydc_discography_options'); echo '<input type="text" name="mydc_discography_options[artist_id]" value="' . esc_attr($options['artist_id'] ?? '') . '" class="regular-text"><p class="description">Für automatischen Import aller Releases des Künstlers</p>'; }
    public function render_color_card_bg() { $options = get_option('mydc_discography_options'); $v = $options['color_card_bg'] ?? '#f5f5f5'; echo '<input type="color" name="mydc_discography_options[color_card_bg]" value="' . esc_attr($v) . '"><input type="text" value="' . esc_attr($v) . '" class="small-text" style="margin-left:10px;" readonly>'; }
    public function render_color_card_title() { $options = get_option('mydc_discography_options'); $v = $options['color_card_title'] ?? '#222222'; echo '<input type="color" name="mydc_discography_options[color_card_title]" value="' . esc_attr($v) . '"><input type="text" value="' . esc_attr($v) . '" class="small-text" style="margin-left:10px;" readonly>'; }
    public function render_color_card_text() { $options = get_option('mydc_discography_options'); $v = $options['color_card_text'] ?? '#666666'; echo '<input type="color" name="mydc_discography_options[color_card_text]" value="' . esc_attr($v) . '"><input type="text" value="' . esc_attr($v) . '" class="small-text" style="margin-left:10px;" readonly>'; }
    public function render_color_card_hover() { $options = get_option('mydc_discography_options'); $v = $options['color_card_hover'] ?? '#000000'; echo '<input type="color" name="mydc_discography_options[color_card_hover]" value="' . esc_attr($v) . '"><input type="text" value="' . esc_attr($v) . '" class="small-text" style="margin-left:10px;" readonly><span class="description" style="margin-left:10px;">Transparenz wird automatisch angewendet</span>'; }
    public function render_color_nocover_bg() { $options = get_option('mydc_discography_options'); $v = $options['color_nocover_bg'] ?? '#dddddd'; echo '<input type="color" name="mydc_discography_options[color_nocover_bg]" value="' . esc_attr($v) . '"><input type="text" value="' . esc_attr($v) . '" class="small-text" style="margin-left:10px;" readonly>'; }
    public function render_color_nocover_text() { $options = get_option('mydc_discography_options'); $v = $options['color_nocover_text'] ?? '#999999'; echo '<input type="color" name="mydc_discography_options[color_nocover_text]" value="' . esc_attr($v) . '"><input type="text" value="' . esc_attr($v) . '" class="small-text" style="margin-left:10px;" readonly>'; }
    public function render_color_modal_bg() { $options = get_option('mydc_discography_options'); $v = $options['color_modal_bg'] ?? '#ffffff'; echo '<input type="color" name="mydc_discography_options[color_modal_bg]" value="' . esc_attr($v) . '"><input type="text" value="' . esc_attr($v) . '" class="small-text" style="margin-left:10px;" readonly>'; }
    public function render_color_modal_text() { $options = get_option('mydc_discography_options'); $v = $options['color_modal_text'] ?? '#222222'; echo '<input type="color" name="mydc_discography_options[color_modal_text]" value="' . esc_attr($v) . '"><input type="text" value="' . esc_attr($v) . '" class="small-text" style="margin-left:10px;" readonly>'; }
    public function render_color_modal_meta() { $options = get_option('mydc_discography_options'); $v = $options['color_modal_meta'] ?? '#666666'; echo '<input type="color" name="mydc_discography_options[color_modal_meta]" value="' . esc_attr($v) . '"><input type="text" value="' . esc_attr($v) . '" class="small-text" style="margin-left:10px;" readonly>'; }
    public function render_color_modal_btn_bg() { $options = get_option('mydc_discography_options'); $v = $options['color_modal_btn_bg'] ?? '#333333'; echo '<input type="color" name="mydc_discography_options[color_modal_btn_bg]" value="' . esc_attr($v) . '"><input type="text" value="' . esc_attr($v) . '" class="small-text" style="margin-left:10px;" readonly>'; }
    public function render_color_modal_btn_text() { $options = get_option('mydc_discography_options'); $v = $options['color_modal_btn_text'] ?? '#ffffff'; echo '<input type="color" name="mydc_discography_options[color_modal_btn_text]" value="' . esc_attr($v) . '"><input type="text" value="' . esc_attr($v) . '" class="small-text" style="margin-left:10px;" readonly>'; }

    public function render_settings_page() {
    // Cache leeren wenn angefordert
    if (isset($_POST['mydc_clear_cache']) && check_admin_referer('mydc_clear_cache_action')) {
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%mydc_discography_auto%' OR option_name LIKE '%mydc_release_thumb%'");
        echo '<div class="notice notice-success"><p>Cache wurde geleert!</p></div>';
    }

    $options = get_option('mydc_discography_options');
    $card_bg = $options['color_card_bg'] ?? '#f5f5f5';
    $card_title = $options['color_card_title'] ?? '#222222';
    $card_text = $options['color_card_text'] ?? '#666666';
    $nocover_bg = $options['color_nocover_bg'] ?? '#dddddd';
    $nocover_text = $options['color_nocover_text'] ?? '#999999';
    $modal_bg = $options['color_modal_bg'] ?? '#ffffff';
    $modal_text = $options['color_modal_text'] ?? '#222222';
    $modal_meta = $options['color_modal_meta'] ?? '#666666';
    $modal_btn_bg = $options['color_modal_btn_bg'] ?? '#333333';
    $modal_btn_text = $options['color_modal_btn_text'] ?? '#ffffff';
    ?>
    <div class="wrap">
    <h1>My Discogs Discography - Einstellungen</h1>

    <!-- Cache leeren Button -->
    <form method="post" style="margin-bottom: 20px;">
    <?php wp_nonce_field('mydc_clear_cache_action'); ?>
    <button type="submit" name="mydc_clear_cache" class="button button-secondary">🗑️ Discogs-Cache leeren</button>
    <span class="description" style="margin-left: 10px;">Löscht gecachte Releases und Cover-Thumbnails</span>
    </form>

    <form method="post" action="options.php">
    <?php
    settings_fields('mydc_discography_settings');
    do_settings_sections('mydc-discography-settings');
    submit_button();
    ?>
    </form>

    <hr>
    <h2>Design-Vorschau</h2>
    <p class="description">So werden die Kacheln und das Modal mit den aktuellen Farbeinstellungen aussehen:</p>

    <div style="display: flex; gap: 40px; flex-wrap: wrap; margin-top: 20px;">
        <!-- Kachel-Vorschau -->
        <div>
            <h4 style="margin-bottom: 10px;">Kachel</h4>
            <div id="mydc-preview-card" style="background: <?php echo esc_attr($card_bg); ?>; padding: 15px; border-radius: 8px; width: 180px; cursor: pointer; transition: transform 0.2s, box-shadow 0.2s;">
                <div style="width: 100%; aspect-ratio: 1; background: <?php echo esc_attr($nocover_bg); ?>; border-radius: 4px; display: flex; align-items: center; justify-content: center; color: <?php echo esc_attr($nocover_text); ?>; font-size: 0.85em;">Kein Cover</div>
                <div style="margin-top: 10px;">
                    <h3 style="margin: 0 0 5px; font-size: 0.95em; line-height: 1.3; color: <?php echo esc_attr($card_title); ?>;">Beispiel Album</h3>
                    <span style="font-size: 0.8em; color: <?php echo esc_attr($card_text); ?>; display: block;">2024</span>
                    <span style="font-size: 0.8em; color: <?php echo esc_attr($card_text); ?>; display: block;">Main</span>
                </div>
            </div>
        </div>

        <!-- Modal-Vorschau -->
        <div>
            <h4 style="margin-bottom: 10px;">Modal</h4>
            <div id="mydc-preview-modal" style="background: <?php echo esc_attr($modal_bg); ?>; padding: 20px; border-radius: 12px; width: 350px; position: relative;">
                <span style="position: absolute; top: 10px; right: 15px; font-size: 24px; color: <?php echo esc_attr($modal_meta); ?>; cursor: pointer;">&times;</span>
                <div style="display: flex; gap: 15px;">
                    <div style="width: 80px; height: 80px; background: <?php echo esc_attr($nocover_bg); ?>; border-radius: 6px; flex-shrink: 0;"></div>
                    <div>
                        <h2 style="margin: 0 0 8px; font-size: 1.1em; color: <?php echo esc_attr($modal_text); ?>;">Beispiel Album</h2>
                        <div style="color: <?php echo esc_attr($modal_meta); ?>; font-size: 0.85em; line-height: 1.6;">
                            <strong>Künstler:</strong> Beispiel Artist<br>
                            <strong>Jahr:</strong> 2024<br>
                            <strong>Label:</strong> Beispiel Label
                        </div>
                    </div>
                </div>
                <a href="#" style="display: inline-block; margin-top: 15px; padding: 8px 16px; background: <?php echo esc_attr($modal_btn_bg); ?>; color: <?php echo esc_attr($modal_btn_text); ?>; text-decoration: none; border-radius: 6px; font-size: 0.85em;">Auf Discogs ansehen →</a>
            </div>
        </div>
    </div>

    <hr style="margin-top: 30px;">
    <h2>Shortcode Verwendung</h2>
    <table class="widefat" style="max-width: 700px;">
    <thead>
    <tr><th>Shortcode</th><th>Beschreibung</th></tr>
    </thead>
    <tbody>
    <tr><td><code>[mydc_discography]</code></td><td>Alle Releases (Auto + Manuell)</td></tr>
    <tr><td><code>[mydc_discography source="auto"]</code></td><td>Nur automatische Discogs-Releases</td></tr>
    <tr><td><code>[mydc_discography source="manual"]</code></td><td>Nur manuell hinzugefügte</td></tr>
    <tr><td><code>[mydc_discography category="gastauftritte"]</code></td><td>Nach Kategorie (Slug)</td></tr>
    <tr><td><code>[mydc_discography role="Appearance"]</code></td><td>Nach Rolle filtern</td></tr>
    <tr><td><code>[mydc_discography filter="true"]</code></td><td>Mit Filter-Dropdown</td></tr>
    <tr><td><code>[mydc_discography orderby="year" order="DESC"]</code></td><td>Sortierung</td></tr>
    </tbody>
    </table>
    </div>
    <?php
    }

    public function ajax_fetch_discogs_release() {
        $release_id = intval($_POST['release_id'] ?? 0);
        $type = sanitize_text_field($_POST['type'] ?? 'release');
        if (!$release_id) { wp_send_json_error('Keine Release-ID'); }
        $options = get_option('mydc_discography_options');
        $token = $options['api_token'] ?? '';
        if (empty($token)) { wp_send_json_error('Kein API Token konfiguriert'); }
        $endpoint = $type === 'master' ? 'masters' : 'releases';
        $url = "https://api.discogs.com/{$endpoint}/{$release_id}";
        $response = wp_remote_get($url, ['headers' => ['Authorization' => 'Discogs token=' . $token, 'User-Agent' => 'MyDCDiscographyPro/2.2'], 'timeout' => 15]);
        if (is_wp_error($response)) { wp_send_json_error($response->get_error_message()); }
        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (isset($data['message'])) { wp_send_json_error($data['message']); }
        wp_send_json_success($data);
    }

    public function ajax_get_release_details() {
        if (isset($_POST['manual_id'])) {
            $post_id = intval($_POST['manual_id']);
            $meta = get_post_meta($post_id);
            $tracklist = [];
            if (!empty($meta['_mydc_tracklist'][0])) {
                foreach (explode("\n", $meta['_mydc_tracklist'][0]) as $line) {
                    $line = trim($line);
                    if ($line) { preg_match('/^(.+?)(?:\s*\((\d+:\d+)\))?$/', $line, $m); $tracklist[] = ['title' => $m[1] ?? $line, 'duration' => $m[2] ?? '', 'type_' => 'track']; }
                }
            }
            wp_send_json_success(['title' => get_the_title($post_id), 'artists' => [['name' => $meta['_mydc_artist'][0] ?? '']], 'year' => $meta['_mydc_year'][0] ?? '', 'labels' => [['name' => $meta['_mydc_label'][0] ?? '']], 'formats' => [['name' => $meta['_mydc_format'][0] ?? '']], 'genres' => [$meta['_mydc_genre'][0] ?? ''], 'images' => [['uri' => $meta['_mydc_cover_url'][0] ?? '']], 'uri' => $meta['_mydc_discogs_url'][0] ?? '', 'tracklist' => $tracklist]);
            return;
        }
        $release_id = intval($_POST['release_id'] ?? 0);
        $type = sanitize_text_field($_POST['type'] ?? 'release');
        if (!$release_id) { wp_send_json_error('Keine Release-ID'); }
        $options = get_option('mydc_discography_options');
        $token = $options['api_token'] ?? '';
        if (empty($token)) { wp_send_json_error('Kein API Token konfiguriert'); }
        $endpoint = $type === 'master' ? 'masters' : 'releases';
        $url = "https://api.discogs.com/{$endpoint}/{$release_id}";
        $response = wp_remote_get($url, ['headers' => ['Authorization' => 'Discogs token=' . $token, 'User-Agent' => 'MyDCDiscographyPro/2.2'], 'timeout' => 15]);
        if (is_wp_error($response)) { wp_send_json_error($response->get_error_message()); }
        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (isset($data['message'])) { wp_send_json_error($data['message']); }
        wp_send_json_success($data);
    }

    private function fetch_auto_releases() {
        $options = get_option('mydc_discography_options');
        $token = $options['api_token'] ?? '';
        $artist_id = $options['artist_id'] ?? '';
        if (empty($token) || empty($artist_id)) { return []; }
        $cache_key = 'mydc_discography_auto_' . $artist_id;
        $cached = get_transient($cache_key);
        if ($cached !== false) { return $cached; }
        $url = "https://api.discogs.com/artists/{$artist_id}/releases?per_page=100";
        $response = wp_remote_get($url, ['headers' => ['Authorization' => 'Discogs token=' . $token, 'User-Agent' => 'MyDCDiscographyPro/2.2'], 'timeout' => 15]);
        if (is_wp_error($response)) { return []; }
        $data = json_decode(wp_remote_retrieve_body($response), true);
        $releases = $data['releases'] ?? [];
        foreach ($releases as &$r) {
            $r['_source'] = 'auto';
            if (empty($r['thumb'])) {
                $detail_cache_key = 'mydc_release_thumb_' . $r['id'];
                $cached_thumb = get_transient($detail_cache_key);
                if ($cached_thumb !== false) { $r['thumb'] = $cached_thumb; }
                else {
                    $endpoint = ($r['type'] === 'master') ? 'masters' : 'releases';
                    $detail_url = "https://api.discogs.com/{$endpoint}/{$r['id']}";
                    $detail_response = wp_remote_get($detail_url, ['headers' => ['Authorization' => 'Discogs token=' . $token, 'User-Agent' => 'MyDCDiscographyPro/2.2'], 'timeout' => 15]);
                    if (!is_wp_error($detail_response)) {
                        $detail_data = json_decode(wp_remote_retrieve_body($detail_response), true);
                        if (!empty($detail_data['images'][0]['uri150'])) { $r['thumb'] = $detail_data['images'][0]['uri150']; }
                        elseif (!empty($detail_data['images'][0]['uri'])) { $r['thumb'] = $detail_data['images'][0]['uri']; }
                        elseif (!empty($detail_data['thumb'])) { $r['thumb'] = $detail_data['thumb']; }
                        set_transient($detail_cache_key, $r['thumb'] ?: '', 30 * DAY_IN_SECONDS);
                    }
                    usleep(300000);
                }
            }
        }
        set_transient($cache_key, $releases, DAY_IN_SECONDS);
        return $releases;
    }

    private function fetch_manual_releases($category = '') {
        $args = ['post_type' => $this->post_type, 'posts_per_page' => -1, 'post_status' => 'publish'];
        if ($category) { $args['tax_query'] = [['taxonomy' => 'mydc_release_category', 'field' => 'slug', 'terms' => $category]]; }
        $posts = get_posts($args);
        $releases = [];
        foreach ($posts as $post) {
            $meta = get_post_meta($post->ID);
            $releases[] = ['id' => $post->ID, 'title' => $post->post_title, 'year' => $meta['_mydc_year'][0] ?? 0, 'thumb' => $meta['_mydc_cover_url'][0] ?? '', 'role' => $meta['_mydc_role'][0] ?? '', 'type' => 'manual', '_source' => 'manual'];
        }
        return $releases;
    }

    public function render_discography($atts) {
        $atts = shortcode_atts(['source' => 'all', 'category' => '', 'role' => '', 'filter' => 'true', 'orderby' => 'year', 'order' => 'DESC', 'limit' => 0], $atts);
        $releases = [];
        if ($atts['source'] === 'all' || $atts['source'] === 'auto') { $releases = array_merge($releases, $this->fetch_auto_releases()); }
        if ($atts['source'] === 'all' || $atts['source'] === 'manual') { $releases = array_merge($releases, $this->fetch_manual_releases($atts['category'])); }
        if (!empty($atts['role'])) { $releases = array_filter($releases, function($r) use ($atts) { return stripos($r['role'] ?? '', $atts['role']) !== false; }); }
        usort($releases, function($a, $b) use ($atts) { $val_a = $a[$atts['orderby']] ?? 0; $val_b = $b[$atts['orderby']] ?? 0; return $atts['order'] === 'DESC' ? $val_b <=> $val_a : $val_a <=> $val_b; });
        if ($atts['limit'] > 0) { $releases = array_slice($releases, 0, (int)$atts['limit']); }
        $roles = array_unique(array_filter(array_column($releases, 'role')));
        sort($roles);
        ob_start();
        ?>
        <div class="mydc-discography-wrapper">
            <?php if ($atts['filter'] === 'true' && count($roles) > 1): ?>
            <div class="mydc-disco-filter">
                <label for="mydc-role-filter">Filtern nach:</label>
                <select id="mydc-role-filter">
                    <option value="">Alle anzeigen</option>
                    <?php foreach ($roles as $role): ?>
                    <option value="<?php echo esc_attr($role); ?>"><?php echo esc_html($role); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <?php endif; ?>
            <div class="mydc-discography">
                <?php foreach ($releases as $release): $cover = $release['thumb'] ?? ''; ?>
                <div class="mydc-disco-item" data-role="<?php echo esc_attr($release['role'] ?? ''); ?>" data-release-id="<?php echo esc_attr($release['id']); ?>" data-type="<?php echo esc_attr($release['type'] ?? 'release'); ?>" data-source="<?php echo esc_attr($release['_source']); ?>">
                    <?php if (!empty($cover)): ?>
                    <img src="<?php echo esc_url($cover); ?>" alt="<?php echo esc_attr($release['title']); ?>" class="mydc-disco-cover">
                    <?php else: ?>
                    <div class="mydc-disco-cover mydc-disco-nocover">Kein Cover</div>
                    <?php endif; ?>
                    <div class="mydc-disco-info">
                        <h3 class="mydc-disco-title"><?php echo esc_html($release['title']); ?></h3>
                        <?php if (!empty($release['year'])): ?><span class="mydc-disco-year"><?php echo esc_html($release['year']); ?></span><?php endif; ?>
                        <?php if (!empty($release['role'])): ?><span class="mydc-disco-role"><?php echo esc_html($release['role']); ?></span><?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <div id="mydc-disco-modal" class="mydc-disco-modal">
                <div class="mydc-disco-modal-content">
                    <span class="mydc-disco-modal-close">&times;</span>
                    <div class="mydc-disco-modal-body"><div class="mydc-disco-modal-loading">Lade Details...</div></div>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    private function hex_to_rgba($hex, $alpha = 1) {
        $hex = ltrim($hex, '#');
        if (strlen($hex) === 3) { $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2]; }
        $r = hexdec(substr($hex, 0, 2)); $g = hexdec(substr($hex, 2, 2)); $b = hexdec(substr($hex, 4, 2));
        return "rgba($r, $g, $b, $alpha)";
    }

    public function enqueue_assets() {
        $options = get_option('mydc_discography_options');
        $card_bg = $options['color_card_bg'] ?? '#f5f5f5';
        $card_title = $options['color_card_title'] ?? '#222222';
        $card_text = $options['color_card_text'] ?? '#666666';
        $card_hover = $options['color_card_hover'] ?? '#000000';
        $nocover_bg = $options['color_nocover_bg'] ?? '#dddddd';
        $nocover_text = $options['color_nocover_text'] ?? '#999999';
        $modal_bg = $options['color_modal_bg'] ?? '#ffffff';
        $modal_text = $options['color_modal_text'] ?? '#222222';
        $modal_meta = $options['color_modal_meta'] ?? '#666666';
        $modal_btn_bg = $options['color_modal_btn_bg'] ?? '#333333';
        $modal_btn_text = $options['color_modal_btn_text'] ?? '#ffffff';
        $hover_shadow = $this->hex_to_rgba($card_hover, 0.15);

        wp_add_inline_style('wp-block-library', "
.mydc-discography-wrapper { max-width: 1200px; margin: 0 auto; }
.mydc-disco-filter { margin-bottom: 20px; }
.mydc-disco-filter select { padding: 8px 12px; font-size: 1em; border-radius: 4px; border: 1px solid #ccc; }
.mydc-discography { display: grid; grid-template-columns: repeat(auto-fill, minmax(180px, 1fr)); gap: 20px; }
.mydc-disco-item { background: {$card_bg}; padding: 15px; border-radius: 8px; cursor: pointer; transition: transform 0.2s, box-shadow 0.2s; }
.mydc-disco-item:hover { transform: translateY(-3px); box-shadow: 0 4px 12px {$hover_shadow}; }
.mydc-disco-item.hidden { display: none; }
.mydc-disco-cover { width: 100%; aspect-ratio: 1; object-fit: cover; border-radius: 4px; }
.mydc-disco-nocover { background: {$nocover_bg}; display: flex; align-items: center; justify-content: center; color: {$nocover_text}; font-size: 0.85em; }
.mydc-disco-info { margin-top: 10px; }
.mydc-disco-title { margin: 0 0 5px; font-size: 0.95em; line-height: 1.3; color: {$card_title} !important; }
.mydc-disco-year, .mydc-disco-role { font-size: 0.8em; color: {$card_text} !important; display: block; }
.mydc-disco-modal { display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.8); z-index: 9999; overflow-y: auto; }
.mydc-disco-modal.active { display: flex; align-items: center; justify-content: center; }
.mydc-disco-modal-content { background: {$modal_bg}; max-width: 700px; width: 90%; max-height: 90vh; overflow-y: auto; border-radius: 12px; position: relative; margin: 20px; }
.mydc-disco-modal-close { position: absolute; top: 15px; right: 20px; font-size: 28px; cursor: pointer; color: {$modal_meta}; z-index: 10; }
.mydc-disco-modal-close:hover { color: {$modal_text}; }
.mydc-disco-modal-body { padding: 30px; color: {$modal_text}; }
.mydc-disco-modal-loading { text-align: center; padding: 40px; color: {$modal_meta}; }
.mydc-modal-header { display: flex; gap: 20px; margin-bottom: 20px; }
.mydc-modal-cover { width: 200px; flex-shrink: 0; border-radius: 8px; }
.mydc-modal-info h2 { margin: 0 0 10px; font-size: 1.4em; color: {$modal_text}; }
.mydc-modal-meta { color: {$modal_meta}; font-size: 0.9em; line-height: 1.8; }
.mydc-modal-tracklist { margin-top: 20px; color: {$modal_text}; }
.mydc-modal-tracklist h4 { margin-bottom: 10px; color: {$modal_text}; }
.mydc-modal-tracklist ol { margin: 0; padding-left: 20px; }
.mydc-modal-tracklist li { padding: 5px 0; border-bottom: 1px solid {$modal_meta}; }
.mydc-modal-discogs-link { display: inline-block; margin-top: 20px; padding: 10px 20px; background: {$modal_btn_bg}; color: {$modal_btn_text}; text-decoration: none; border-radius: 6px; font-size: 0.9em; }
.mydc-modal-discogs-link:hover { opacity: 0.9; color: {$modal_btn_text}; }
@media (max-width: 500px) { .mydc-modal-header { flex-direction: column; } .mydc-modal-cover { width: 100%; } }
");

        wp_add_inline_script('jquery', '
jQuery(document).ready(function($) {
    $("#mydc-role-filter").on("change", function() {
        var role = $(this).val();
        $(".mydc-disco-item").each(function() {
            if (role === "" || $(this).data("role") === role) { $(this).removeClass("hidden"); } else { $(this).addClass("hidden"); }
        });
    });
    $(".mydc-disco-item").on("click", function() {
        var releaseId = $(this).data("release-id");
        var type = $(this).data("type");
        var source = $(this).data("source");
        var modal = $("#mydc-disco-modal");
        modal.addClass("active");
        $(".mydc-disco-modal-body").html("<div class=\"mydc-disco-modal-loading\">Lade Details...</div>");
        var postData = { action: "mydc_get_release_details" };
        if (source === "manual") { postData.manual_id = releaseId; } else { postData.release_id = releaseId; postData.type = type; }
        $.post("' . admin_url('admin-ajax.php') . '", postData, function(response) {
            if (response.success) {
                var d = response.data;
                var html = "<div class=\"mydc-modal-header\">";
                if (d.images && d.images[0] && d.images[0].uri) { html += "<img src=\"" + d.images[0].uri + "\" class=\"mydc-modal-cover\">"; }
                html += "<div class=\"mydc-modal-info\">";
                html += "<h2>" + (d.title || "Unbekannt") + "</h2>";
                html += "<div class=\"mydc-modal-meta\">";
                if (d.artists && d.artists[0] && d.artists[0].name) html += "<strong>Künstler:</strong> " + d.artists.map(a => a.name).join(", ") + "<br>";
                if (d.year) html += "<strong>Jahr:</strong> " + d.year + "<br>";
                if (d.labels && d.labels[0] && d.labels[0].name) html += "<strong>Label:</strong> " + d.labels[0].name + "<br>";
                if (d.formats && d.formats[0] && d.formats[0].name) html += "<strong>Format:</strong> " + d.formats[0].name + "<br>";
                if (d.genres && d.genres[0]) html += "<strong>Genre:</strong> " + d.genres.filter(g => g).join(", ") + "<br>";
                html += "</div></div></div>";
                if (d.tracklist && d.tracklist.length) {
                    html += "<div class=\"mydc-modal-tracklist\"><h4>Tracklist</h4><ol>";
                    d.tracklist.forEach(function(t) { if (t.type_ === "track") { html += "<li>" + t.title + (t.duration ? " <small>(" + t.duration + ")</small>" : "") + "</li>"; } });
                    html += "</ol></div>";
                }
                if (d.uri) { html += "<a href=\"" + d.uri + "\" target=\"_blank\" class=\"mydc-modal-discogs-link\">Auf Discogs ansehen</a>"; }
                $(".mydc-disco-modal-body").html(html);
            } else { $(".mydc-disco-modal-body").html("<p>Fehler beim Laden: " + (response.data || "Unbekannter Fehler") + "</p>"); }
        }).fail(function() { $(".mydc-disco-modal-body").html("<p>Verbindungsfehler. Bitte versuche es erneut.</p>"); });
    });
    $(".mydc-disco-modal-close, .mydc-disco-modal").on("click", function(e) { if (e.target === this) { $("#mydc-disco-modal").removeClass("active"); } });
    $(document).on("keydown", function(e) { if (e.key === "Escape") $("#mydc-disco-modal").removeClass("active"); });
});');
    }
}

new MyDC_Discography_Pro();
